///////////////////////////////////////////////////////////////////////////////////////////////
// Created on Mon Apr 22 10:28:59 2024
//
// @author: Andrei Sontag
//
// This code simulates the model proposed by Dodd et al. (see reference) for different
// values of neighbourhood size and recruitment probability. The generated data is a 
// count of the number of times the system has a defined number of methylated and acetylated
// nucleosomes in the population during the simulation. This is saved in a folder called 'nuc'.
// It also computes the "polarisation" (z) of the population, saving it as another file.
// The data is then analysed using the script 'nucleosome_analysis.py'.
//
// References:
// Dodd IB, Micheelsen MA, Sneppen K, Thon G. Theoretical analysis of epigenetic cell 
// memory by nucleosome modification. Cell. 2007 May 18;129(4):813-22. 
// doi: 10.1016/j.cell.2007.02.053. PMID: 17512413.
//
///////////////////////////////////////////////////////////////////////////////////////////////

#include <iostream>
#include <cmath>
#include <random>
#include <chrono>
#include <fstream>
#include <vector>
#include <algorithm>

using namespace std;
using namespace std::chrono;

auto start = high_resolution_clock::now();

int main() {
    random_device rd;
    mt19937 gen(rd());

    const int N = 60;
    const int dn = 1;
    const double dp = 0.01;

    vector<int> nvals(61);
    generate(nvals.begin(), nvals.end(), [n = 0, &dn]() mutable { return n++ * dn; });
    vector<double> pvals(101);
    generate(pvals.begin(), pvals.end(), [n = 0, &dp]() mutable { return n++ * dp; });

    uniform_real_distribution<> disuni(0, 1);

    const int t_max = 1000000;

    for (int m=0; m<61; m++) // loop through avals, check size of avals
    {
        cout << "completed:" << m * 100.0/ 31.0 << "%" << endl;
        for (int h = 0; h < 101; h++) // loop through bvals, check size of bvals
        {
            ofstream myfile_dir ("./nuc/nuc_"+to_string(m)+"_"+to_string(h)+".txt");
            ofstream myfile_z ("./nuc/z_"+to_string(m)+"_"+to_string(h)+".txt");

            double p = pvals[h];
            int neigh = nvals[m];

            cout << p << ',' << neigh << endl;

            int count[N+1][N+1] = {0};
            int z[2*N+1] = {0};
            int s[N] = {0}; // starts as methylated

            for (int t = 0; t<t_max; t++) {
                int focus_particle = floor(disuni(gen) * N);

                double rand = disuni(gen);

                // step 2A
                if (rand <= p) {
                    int neigh_min = max(0, focus_particle - neigh);
                    int neigh_max = min(N - 1, focus_particle + neigh);

                    int partner = floor(disuni(gen) * (neigh_max - neigh_min)) + neigh_min;
                    if (s[partner] == 0) {
                        s[focus_particle] = max(0, s[focus_particle] - 1);

                    } else if (s[partner] == 2) {
                        s[focus_particle] = min(2, s[focus_particle] + 1);
                    }
                }

                // step 2B
                else {
                    double u = disuni(gen);
                    if (u <= 1.0 / 3.0) {
                        s[focus_particle] = max(0, s[focus_particle] - 1);
                    } else if (u <= 2.0 / 3.0) {
                        s[focus_particle] = min(2, s[focus_particle] + 1);
                    }
                }

                int tM = 0, tA = 0;
                for (int i : s) {
                    if (i==0) {
                        tM++;
                    } else if (i==2) {
                        tA++;
                    }
                }

                count[tM][tA]++;
                z[tM-tA+N]++;
            }
            if (myfile_dir.is_open()) {
                //cout << "Completed " << floor(100*t/t_max) << "%. Elapsed time: " << duration.count() << endl;
                for (auto & i : count) {
                    for (int j=0; j<N; j++) {
                        myfile_dir << i[j] << ',';
                    }
                    myfile_dir << i[N] << endl;
                }
            }
            myfile_dir.close();

            if (myfile_z.is_open()) {
                //cout << "Completed " << floor(100*t/t_max) << "%. Elapsed time: " << duration.count() << endl;
                for (int j=0; j<2*N; j++) {
                    myfile_z << z[j] << ',';
                }
                myfile_z << z[2*N] << endl;
            }
            myfile_z.close();
        }
    }

    auto stop = high_resolution_clock::now();
    auto duration = duration_cast<microseconds>(stop - start);

    cout << duration.count()/1000000.0 << endl;
    return 0;
}
